// demo-callbacks.js

// This file contains your javascript callback functions.
// For a basic overview of how this works see any tutorial or
// introduction to asynchronous javascript and HTML.

// Edit this file to implement your HTML UI.

// When you use send_event(), it will return immediately. When the 
// native view is finished processing the event, it will send a response
// back. The functions in bottlenose.js read that response and 
// then automatically call one of these "callback" functions.
// They are called callbacks because the server "calls back" when
// it has the response ready.
//
// Every callback function must be named in a precise way. The
// name must be "event_" + eventID. So if your event ID is "play_pause"
// then the callback function must be named event_play_pause(). If
// the name is wrong then the response will fail to connect.
//
// Your callback should have a single argument "r". This is the 
// response from the server in the form of a dictionary (aka hashmap).
// You can read out of "r" using the function get_value().
//
// "r" is a set of key/value pairs. The first value you should always
// read has the key kBottleKeyResponseCode. This is the response code
// from the server side (from the view actually) and it will be 
// equal to kBottleEventOK if your event worked. You access the key
// using the get_value() function, which must have "r" as the first
// variable and the key that you want to access (such as 
// kBottleKeyResponseCode) as the second parameter.
// So call get_value(r, kBottleKeyResponseCode) .
// In general call get_value(r, "myKey")
//
// The return dictionary will also always include all of the parameters
// that you sent with your event call, so that you can figure out 
// which callback was associated with which event.
//
// The native view programmer determines what return values are
// provided.


// Callbacks for LiveVideoMixer


// Play or Pause - event callback
// This function is called when the native view is done playing or pausing the video.
// Doesn't return any special values other than the response code
function event_play_pause(r)
{
  // Check for any errors
  if( get_value(r, kBottleKeyResponseCode) != kBottleEventOK ) { show_error( "event_play_pause error: " + get_value(r, kBottleKeyResponseCode) + "."); return; }
  
  // By the time this callback is called, the native view is already playing
  // so we should flip the global "is playing" bit now
//  mPlaying = !mPlaying;
  
  // Toggle the PLAY/STOP button
  if( ! mPlaying ) {
    start_play();
  }
  else {
    stop_play();
  }
}


// Edit a Channel - event callback
// This function is called when the native view is done setting a channel source video
// Returns:
// "channel" - the channel that was set
// "filePath" - the full path of the source file
function event_set_source(r)
{
  // The user can cancel opening a file on the native side.
  if( get_value(r, kBottleKeyResponseCode) == kBottleEventCancel  ) {
    show_status("Cancelled set source.");
    return;
  }
  // Check for any others errors
  if( get_value(r, kBottleKeyResponseCode) != kBottleEventOK ) { show_error( "event_set_source error: " + get_value(r, kBottleKeyResponseCode) + "."); return; }
  
  // When a source is set, the native view will stop playback
  if( mPlaying ) {
    stop_play();
  }
  
  // use get_value() to find out what channel it is
  var ch = get_value(r, kKeyChannel);

  // enable the channel and set up the UI if this is the first channel
  enable_channel(ch);
}


// Edit a Channel - event callback
// This function is called when the native view is done setting up a channel for editing.
// Editing a channel has an effect both on the native view (it allows dragging and resizing)
// and on the HTML UI (it enabled changing the mask and the opacity).
// Returns:
// "channel" - the channel that was clicked
// You have to keep track of which channel is currently being edited yourself.
function event_edit_channel(r)
{
  // The user may have tried to edit a channel that has no video yet
  if( get_value(r, kBottleKeyResponseCode) == kBottleEventOutOfRange ) {
    show_error( "Attempted to edit a channel (" + get_value(r, kKeyChannel) + ") that has no source." );
    return;
  }
  // Check for any others errors
  if( get_value(r, kBottleKeyResponseCode) != kBottleEventOK ) { show_error( "event_edit error: " + get_value(r, kBottleKeyResponseCode) + "."); return; }
  
  // use get_value() to find out what channel they are editing
  var ch = get_value(r, kKeyChannel);
  
  // if it's the current channel, we actually toggle off
  if( ch == mCurrentCh ) {
    unset_edit_channel(ch); // toggle off
    mCurrentCh = -1; // remember no channel being edited now
    return; // that's all
  }
  
  // is there a channel currently being edited?
  if( mCurrentCh != -1 )
    unset_edit_channel(mCurrentCh); // updates the UI
  // remember what channel is being edited
  mCurrentCh = ch;
  
  // update the UI
  set_edit_channel(ch);
}


// Set Mask - event handler
// This function is called when the native view is done changing the mask for a channel.
// Note that this always affects the current editing channel.
// Returns:
// "mask" - the number of the mask that was chosen.
function event_set_mask(r)
{
  // Not much here. Check for errors, print a status message.
  if( get_value(r, kBottleKeyResponseCode) != kBottleEventOK ) { show_error( "Error in event_mask: response code: " + get_value(r, kBottleKeyResponseCode) + "."); return; }
  show_status( "Set the mask for channel " + mCurrentCh + ".");
}


// Set Opacity - event handler
// This function is called when the native view is done setting the opacity for a channel.
// Returns:
// "opacity" - the value between 0 and 100 that you sent in
function event_set_opacity(r)
{
  // The user may have entered a value that's out of the allowed range
  if( get_value(r, kBottleKeyResponseCode) == kBottleEventOutOfRange ) {
    show_status( "Opacity must be between 0 and 100." );
    return;
  }
  // any other errors?
  if( get_value(r, kBottleKeyResponseCode) != kBottleEventOK ) { show_error( "Error in event_set_opacity: response code: " + get_value(r, kBottleKeyResponseCode) + "."); return; }
  
  // print some status
  show_status( "Set opacity in channel " + mCurrentCh + " to " + get_value(r, "opacity") + ".");
}


