//
//  FractalView.m
//  FractalTrees
//
//  Created by Simon Woodside on Thurs May 30 2002.
//  Copyright (c) 2002 Simon Woodside. All rights reserved.
//

#import "FractalView.h"
#import "FractalTreeGrower.h"
#import "FractalTreeBranch.h"
#import "ColourController.h"
#import "FractalController.h"
#import "RandSupply.h"

#define DEBUG_GRAPHICS NO
#define SW_INITIAL_LINE_CAP_STYLE NSRoundLineCapStyle
#define SW_SCALE_TO_FIT_PADDING 0
#define SW_ABORT_INTERVAL 0.2
//0.033
#define SW_FLUSH_INTERVAL 0.25
#define SW_BASE_LENGTH 20

@implementation FractalView

// Various NSRect utility functions
- (NSRect)padRect:(NSRect)rect withPadding:(float)padding;
{
    // Make a copy
    NSRect result;
    result.origin.x = rect.origin.x - padding;
    result.origin.y = rect.origin.y - padding;
    result.size.width = rect.size.width + 2 * padding;
    result.size.height = rect.size.height + 2 * padding;
    return result;
}

- (NSRect)fitRectToAspectRatio:(NSRect)rect
{
    float heightShouldBe;
    float widthShouldBe;
    float difference;
    NSRect paddedRect = [self padRect:rect
                          withPadding:baseLength*lengthFactor*widthFactor/1.75];
    // the 1.0 is a total hack. the real problem is that we need to pad
    // each time we update the autofit rect. then use 2 here should be right.
    NSRect result = NSOffsetRect(paddedRect, 0.0, 0.0); // cheap hack
                                                        // cause I don't know how to
                                                        // properly copy a struct

    //    printf("SWFitRectToAspectRatio: oldRect = %s ",
    //           [NSStringFromRect(rect) cString]);
    //    printf("oldAspectRatio = %f ", rect.size.width/rect.size.height );
    //    printf("masterAspectRatio = %f\n", aspectRatio );

    //fix the aspect ratio by expanding the rect
    heightShouldBe = paddedRect.size.width / aspectRatio;
    widthShouldBe = paddedRect.size.height * aspectRatio;
    if( heightShouldBe >= paddedRect.size.height )
    {
        difference = (heightShouldBe - paddedRect.size.height);
        result.size.height += difference;
        result.origin.y -= difference/2.0;
    } else {
        difference = (widthShouldBe - paddedRect.size.width);
        result.size.width += difference;
        result.origin.x -= difference/2.0;
    }

    printf("SWFitRectToAspectRatio: newRect = %s\n",
           [NSStringFromRect(result) cString]);
    //    printf("newAspectRatio = %f\n", result.size.width/result.size.height );

    return result;
}


// p2 - p1
- (NSPoint)subtractPoint:(NSPoint)p1 fromPoint:(NSPoint)p2;
{
    return NSMakePoint( p2.x - p1.x, p2.y - p1.y );
}
- (NSPoint)addPoint:(NSPoint)p1 toPoint:(NSPoint)p2;
{
    return NSMakePoint( p2.x + p1.x, p2.y + p1.y );
}


// **************************************************************
// Init
// **************************************************************
//
// **************************************************************
// Initialize the view
//
- (id)initWithFrame:(NSRect)frame
{
    NSRect cursorRect;
    NSPoint midSpot;
    NSPrintInfo * sharedPrintInfo;
    
    [super initWithFrame:frame];
    
    reEnableAutofit = NO;
    
    // Grab hand cursor
    cursorRect = NSMakeRect(0,0,16,16);
    midSpot = NSMakePoint(NSMidX(cursorRect),NSMidY(cursorRect));
    openGrabHandCursor = [[NSCursor alloc] 
        initWithImage:[NSImage imageNamed:@"OpenGrabHandCursor.tiff"] hotSpot:midSpot];
    closedGrabHandCursor = [[NSCursor alloc] 
        initWithImage:[NSImage imageNamed:@"ClosedGrabHandCursor.tiff"] hotSpot:midSpot];
    mouseIsDragging = NO;

    // Set some reasonable defaults for the initial tree (somewhat pointless with autofit)
    basePoint = NSMakePoint(0, 0);
    baseLength = SW_BASE_LENGTH; //this is small for the workaround in copy
    
    // How annoying, we can't access UI controls from this method
    haveSetDefaults = NO;
    
    // Autofit defaults
    allowAutofit = YES;
    reEnableAutofit = NO;
    autofitNextTime = NO;
    lastDrawWasAborted = NO;
    
    shouldDrawVisible = YES;
    
    // Draw stuff
    [NSBezierPath setDefaultLineCapStyle:SW_INITIAL_LINE_CAP_STYLE];
    lastFlushDate = [[NSDate alloc] initWithTimeIntervalSinceNow:0];
    lastAbortDate = [[NSDate alloc] initWithTimeIntervalSinceNow:0];
    drawFlushFreq = EveryLevel;
    
    // Send reasonable settings for printing to the shared NSPrintInfo object
    sharedPrintInfo = [NSPrintInfo sharedPrintInfo];
    [sharedPrintInfo setHorizontalPagination:NSFitPagination];
    [sharedPrintInfo setVerticalPagination:NSFitPagination];
    [sharedPrintInfo setHorizontallyCentered:YES];
    [sharedPrintInfo setVerticallyCentered:YES];
    [NSPrintInfo setSharedPrintInfo:sharedPrintInfo];
    
    // Grow the initial tree
    [grower newTree]; //growTree];
    
    return self;
}


// **************************************************************
// **************************************************************
// Draw
//
// **************************************************************
// Draw the view (called whenever it is necessary)
// RECURSIVE: note this may be called recursively for autofit!
//
- (void)drawRect:(NSRect)rect;
{
    printf("drawRect: started\n{\n");
    printf("    with rect = %s\n", [NSStringFromRect(rect) cString]);
    printf("    with bounds = %s\n", [NSStringFromRect([self bounds]) cString]);
    printf("    with height = %i\n", [controller getHeight]);
    if( recursing ) { printf("   Inside a recursive call\n"); }
    if( !haveSetDefaults ) { [self initForFirstDraw]; haveSetDefaults = YES; }

    // predict shouldDrawVisible
    shouldDrawVisible = recursing || !allowAutofit;

    //wipe background
    if( shouldDrawVisible )
    { [[colour getBackgroundColour] set]; NSRectFill([self bounds]); }

    //draw tree
    [self resetForDraw];
    [self linearDraw:[grower tree]];
    
    if( abortedDrawForNewEvent ) { allowAutofit = NO; }
    if( recursing ) { allowAutofit = NO; }
    if( allowAutofit && !recursing )  //normally controller sets allowAutofit
    {
        if( [self shouldAutofitBounds] || !shouldDrawVisible )
        {
            // sometimes shouldDrawVisible is predicted wrong!
            printf("    drawRect: Allowed to do & should do an autofit.\n");
            [self autofitBounds];
            recursing = YES;
            [self display]; //draw again but just once from above line
        }
    }
    if( !recursing ) { [controller setProgressIndicatorAnimate:NO]; }
    //This HAS to be at the end of drawRect:
    allowAutofit = NO; recursing = NO; shouldDrawVisible = YES;
    printf("} drawRect: ended\n");
}
- (BOOL)shouldAutofitBounds;
{
    if( NSEqualRects([self bounds], [self fitRectToAspectRatio:newBounds]) )
    { // watch above for padding!!!!
        return NO;
    } else {
        return YES;
    }
}
//
// **************************************************************
// Scale to fit if it's allowed
//
- (void)autofitBounds;
{
    newBounds = [self fitRectToAspectRatio:newBounds];
    [self setBounds:newBounds];
}
//
// **************************************************************
// Draw a single branch
// we do a lot of checking here to put the user in control
//
- (void)drawBranch:(FractalTreeBranch *)branch level:(int)level;
{
    float branchWidth;
    if( shouldDrawVisible )
    {
        // if it's in the view's bounds draw the branch
        if( NSIntersectsRect([branch normalizedRect], [self bounds]) )
        {
            branchWidth = branch->curLength * widthFactor;
//            branchWidth = SW_BASE_LENGTH * lengthFactor * widthFactor;
            branchWidth -= branchWidth * 0.4 * ((level*1.0)/([controller getHeight]*1.0));
            [[colour getColourForLevel:level withPosition:branch->curPosition] set];
            [NSBezierPath setDefaultLineWidth:branchWidth];
            [NSBezierPath strokeLineFromPoint:branch->curBase toPoint:branch->curEnd];
        }

        // if the user has generated a new event, abort this draw cycle
        if( [self shouldAbortBecauseParametersChanged] )
        {
            abortedDrawForNewEvent = YES;
            allowAutofit = NO; // not now, for some reason
        }

        // if "it's time to do it" flush the graphics
        if( [self shouldFlushGraphicsAfterBranch:branch] )
        {
            [[NSGraphicsContext currentContext] flushGraphics];
        }
    }
    
    // if the branch is outside the tree's autofit bounds, update it
    if( ! NSContainsRect(newBounds, [branch normalizedRect]) )
    { 
        if( DEBUG_GRAPHICS ) 
        { 
            [[NSColor yellowColor] set];
            [NSBezierPath strokeRect:[branch normalizedRect]]; 
        }
        newBounds = NSUnionRect(newBounds, [branch normalizedRect]);
    }
}
//
// **************************************************************
// Tells if the user changed the params
//
- (BOOL)shouldAbortBecauseParametersChanged;
{
    if( fabs([lastAbortDate timeIntervalSinceNow]) > SW_ABORT_INTERVAL )
    {
        // tempus fugit, see if we should abort (sorta like 30fps)
        // this left mouse.
        NSEvent * mouseEvent =
        [[self window] nextEventMatchingMask:
            NSLeftMouseDraggedMask | NSLeftMouseDown
                                   untilDate:[NSDate dateWithTimeIntervalSinceNow:0]
                                      inMode:NSEventTrackingRunLoopMode dequeue:NO];
        if( mouseEvent )
        {
            printf( "userChangedParameters: YES.\n" );
            [lastAbortDate initWithTimeIntervalSinceNow:0];
            //fool the flusher into flushing quickly for nice HI ;-)
            [lastFlushDate initWithTimeIntervalSinceNow:0];
            return YES;
        }
    }
    return NO;
}
//           
// **************************************************************
// Tells if it's time to flush the graphics
//
- (BOOL)shouldFlushGraphicsAfterBranch:(FractalTreeBranch *)branch;
{
    switch (drawFlushFreq)
    {
        case EveryBranch:
            return YES;
            break;
        case EveryLevel:
            if( branch->curPosition == 0 )
            { return YES; }
            break;
        case Timed:
            if( fabs([lastFlushDate timeIntervalSinceNow]) > SW_FLUSH_INTERVAL )
            {
                // tempus fugit, flush the graphics context
                [lastFlushDate initWithTimeIntervalSinceNow:0];
                return YES;
            }
            break;
        case Never:
            break;
    }
    return NO;
}
//
// **************************************************************
// Init stuff before the first time drawing
//
- (void)initForFirstDraw;
{
    // Grab initial settings from UI (HACK initialization problem)
    [controller initStuff];
    [controller updateGrowingSettingsFromUI];
    [controller updateDrawingSettingsFromUI];
    [controller updateInformationPanel];
    [grower growTree];
    return;
}
//
// **************************************************************
// Reset globals needed before drawing
//
- (void)resetForDraw;
{
    abortedDrawForNewEvent = NO;
//    drawFlushFreq = Timed;
    drawFlushFreq = [controller getDrawStyle];
    [lastFlushDate initWithTimeIntervalSinceNow:0];
    [lastAbortDate initWithTimeIntervalSinceNow:0];
    aspectRatio = [self bounds].size.width / [self bounds].size.height;
    pixelSize = [self bounds].size.height / [self frame].size.height;
    printf("    resetForDraw: pixelSize = %f\n", pixelSize);
    newBounds = NSRectFromString(@"{{0, 0}, {0, 0}}");
}
//
// **************************************************************
// Draw the tree with linear algorithm
//
- (void)linearDraw:(NSArray *)tree
{
    int level, index, parentIndex;
    FractalTreeBranch * curBranch, * parentBranch;
    NSArray * parentArray;

    printf("    linearDraw: started\n    {\n");
    //draw the base branch specially
//    printf("linearDraw: Pre-baseLength = %f  Pre-lengthFactor = %f  * = %f\n",
//           baseLength, lengthFactor, baseLength * lengthFactor);
    curBranch = [[tree objectAtIndex:0] objectAtIndex:0];
    [curBranch setCurLength:baseLength * lengthFactor
                   curAngle:M_PI_2
                curPosition:0.5
                    curBase:basePoint];
    [curBranch setCurEnd:NSMakePoint(curBranch->curBase.x,
                                     curBranch->curBase.y + curBranch->curLength)];
    newBounds = [curBranch normalizedRect];
    [self drawBranch:curBranch level:0];
    //end of draw base branch

    //linear loop through levels, indices, to drawBranch on all branches in the array
    for( level=1; level<[controller getHeight]; level++ )
    {
//        printf("        linearDraw: starting level = %i\n", level);
        // loop through branches at this level
        for( index=0; index<pow(2,level); index++ )
        {
            if( abortedDrawForNewEvent )
            {
                printf("    } linearDraw: aborted\n");
                return;
            }

            //begin trig and stuff
            //
            parentArray = [tree objectAtIndex:level-1];
            parentIndex = floor(index/2); //set up next 2 lines
            parentBranch = [parentArray objectAtIndex:parentIndex];
            curBranch = [[tree objectAtIndex:level] objectAtIndex:index]; // silly syntax
            [curBranch
                setCurLength:(parentBranch->curLength * curBranch->length)*lengthFactor
                    curAngle:(parentBranch->curAngle + curBranch->angle)];
            if( index%2 ) { curBranch->curAngle -= angleTerm; }
            else { curBranch->curAngle += angleTerm; }
            // reverse next two lines for an interesting effect
            [curBranch setCurBase:parentBranch->curEnd];
            [curBranch setCurEnd:NSMakePoint( 
                curBranch->curBase.x + (curBranch->curLength * cos(curBranch->curAngle)),
                curBranch->curBase.y + (curBranch->curLength * sin(curBranch->curAngle))
                )];
            //
            //end trig and stuff
            
            [curBranch setPositionForIndex:index maxLevel:level];
            [self drawBranch:curBranch level:level];
        }
    }
    // if abort happens, won't get here!
    printf("    } linearDraw: completed (no abort)\n");
}


// **************************************************************
// Drag
// **************************************************************
//
// **************************************************************
// Mouse Events for dragging around the view
// GRRR.. something wrong with this. **** ????
//
- (void)mouseDown:(NSEvent *)event 
{
//    [controller setAllowScaleToFit:NO]; FIX
    NSPoint mouseLocationInWindow = [event locationInWindow];
    mouseOriginInView =
        [self convertPoint:mouseLocationInWindow fromView:[[self window] contentView]];
    boundsOriginInView = [self frame].origin;
    boundsOriginInView.y *= pixelSize; // TOTAL HACK
    //I don't really understand why that works or what's broken.
//        [self convertPoint:[self frame].origin fromView:[[self window] contentView]];
    [closedGrabHandCursor set];
    mouseIsDragging = YES;
}
- (void)mouseDragged:(NSEvent *)event 
{
    [self moveView:[event locationInWindow]];
}
- (void)moveView:(NSPoint)mouseLocationInWindow;
{
    NSPoint mouseLocationInView = 
        [self convertPoint:mouseLocationInWindow fromView:[[self window] contentView]];
    NSPoint mouseDeltaInView =
        [self subtractPoint:mouseLocationInView fromPoint:mouseOriginInView];
    [self translateOriginToPoint:
        [self subtractPoint:mouseDeltaInView fromPoint:boundsOriginInView]];
    [self setNeedsDisplay:YES];
}
- (void)mouseUp:(NSEvent *)event;
{
//    [controller setAllowScaleToFit:YES];  FIX!!!
    mouseIsDragging = NO;
    [[self window] invalidateCursorRectsForView:self];
}


// **************************************************************
// API
// **************************************************************
//
// **************************************************************
// Sets for Tree Display
//
- (void)setAllowAutofit:(BOOL)set;
{
    allowAutofit = set;
}
- (void)setDrawFlushFrequency:(DrawFlushFrequency)freq;
{
    drawFlushFreq = freq;
}
- (void)setLengthFactor:(id)sender
{
    float senderFactor = pow([sender floatValue],0.5);
        lengthFactor = senderFactor;//pow([sender floatValue],0.5);
}
- (void)setAngleTerm:(id)sender
{
    float senderTerm = M_PI*[sender floatValue]; // can't just test it.
        angleTerm = senderTerm;
}
- (void)setWidthFactor:(id)sender
{
        if( [sender floatValue] < 0.001 )
        { widthFactor = 0.001; }
        else { widthFactor = [sender floatValue]; }
}
- (BOOL)abortedLastDrawForNewEvent;
{
    return abortedDrawForNewEvent;
}

// **************************************************************
// Sets for Center
//
/*- (void)centerOriginInView
{
    NSPoint newOrigin;
    newOrigin.x = -[self bounds].size.width/2;
    newOrigin.y = -[self bounds].size.height/2;
    [self setBoundsOrigin:newOrigin];
}*/


// **************************************************************
// Data export
// **************************************************************
//
// **************************************************************
// Copy to clipboard
//
- (IBAction)copy
{
    // workaround: this only works if baseLength is set to a small number
    // basically this scheme fails if "pixelSize" (i.e. the ratio between
    // bounds and frame) ever exceeds 1.0
    NSRect copyRect = [self frame];
    copyRect.origin = [self bounds].origin;

    printf("copy: copyRect = %s\n", [NSStringFromRect(copyRect) cString] );
    
    NSImage * image = [[NSImage alloc] initWithData:
        [self dataWithPDFInsideRect:copyRect]];
    
    NSPasteboard * pboard = [NSPasteboard generalPasteboard];
    [pboard declareTypes:[NSArray arrayWithObject:NSTIFFPboardType] owner:self];
    if( ![pboard setData:[image TIFFRepresentation] forType:NSTIFFPboardType] )
    {
        NSBeep();	// complain
    }
}

// **************************************************************
// Export to TIFF
//
- (IBAction)export
{
    NSSavePanel * panel = [NSSavePanel savePanel];
    [panel setRequiredFileType:@"tiff"];

    [panel beginSheetForDirectory:nil 
        file:@"Fractal Tree.tiff" 
        modalForWindow:[self window] 
        modalDelegate:self 
        didEndSelector:@selector(exportPanelDidEnd: returnCode: contextInfo:) 
        contextInfo:nil];
}
// **************************************************************
// Callback
//
- (void)exportPanelDidEnd:(NSWindow *)sheet 
               returnCode:(int)returnCode
              contextInfo:(void *)contextInfo
{
    NSSavePanel * panel;
    panel = (NSSavePanel *)sheet;
    
//    [sheet orderOut:self]; //dismiss
    
    if( returnCode == NSOKButton )
    {
        NSRect copyRect = [self frame];
        copyRect.origin = [self bounds].origin;

        NSImage * image =
            [[NSImage alloc] initWithData:[self dataWithPDFInsideRect:copyRect]];
        NSData * data = [image TIFFRepresentation];
        
        if (![data writeToFile:[panel filename] atomically:YES])
        {
            NSBeep();
        }
    }
}





// **************************************************************
// Superclass override methods
// **************************************************************
//
// **************************************************************
// Overrides
//
- (void)dealloc
{
    [super dealloc];
}
- (BOOL)isOpaque
{
    return YES;
}
- (BOOL)acceptsFirstResponder
{
    return YES;
}
- (void)resetCursorRects;
{
    if( mouseIsDragging )
    {
        [self addCursorRect:[self visibleRect] cursor:closedGrabHandCursor];
        [closedGrabHandCursor setOnMouseEntered:YES];
    } else {
        [self addCursorRect:[self visibleRect] cursor:openGrabHandCursor];
        [openGrabHandCursor setOnMouseEntered:YES];
    }
}
@end
