<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet
  version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:dyn="http://exslt.org/dynamic"
  extension-element-prefixes="dyn"
  >
  
  <xsl:output 
    method = "xml"
    indent = "yes"
    encoding = "UTF-8"
    />
  <xsl:preserve-space elements="text"/>
  
  <!--**************************************************************-->
  <!--  Alexandra is copyright (C) 2002-3 Simon Woodside.
        Licensed under the GPL - see LICENSE for details -->
  <!--  rng2rngsub
        Concatenate a RNG (Relax NG) schema and start at an
        arbitrary point in the schema defined by an XPath into
        either the Schema (start_at_rng) or into an instance that
        follows the schema (start_at_inst) -->
  <!--**************************************************************-->
  
  <!--**************************************************************-->
  <!-- Parameters                                                   -->
  <!--**************************************************************-->
  
  <xsl:param name="start_at_rng"/>
  <xsl:param name="start_at_inst"/>

  <!--**************************************************************-->
  <!-- Variables                                                    -->
  <!--**************************************************************-->

  <!--**************************************************************-->
  <!--  The XPath to the starting location in the RNG grammar.
        (As opposed to in the instance document.) -->
  <!--  This is guaranteed to have a useful value -->
  <xsl:variable name="StartAtRNGString">
    <xsl:choose>
      <xsl:when test="$start_at_inst != ''"><!-- and $start_at_inst != '/resume'">-->
        <xsl:message>*** StartAtRNGString: Converting from $start_at_inst</xsl:message>
        <xsl:call-template name="inst2rngPath">
          <xsl:with-param name="fragment" select="$start_at_inst"/>
        </xsl:call-template>
      </xsl:when> 
      <xsl:when test="$start_at_rng != ''"> <!-- use RNG start path -->
        <xsl:message>*** StartAtRNGString: Using $start_at_rng</xsl:message>
        <xsl:value-of select="$start_at_rng"/>
      </xsl:when> 
      <xsl:otherwise> <!-- fall back on just / -->
        <xsl:message>*** StartAtRNGString: Falling back on /grammar/start</xsl:message>
        <xsl:text>/grammar/start</xsl:text>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  
  <xsl:variable name="StartAtRNGNodeSet" select="dyn:evaluate($StartAtRNGString)"/>
  
  <!--**************************************************************-->
  <!--  The XPath to the starting location in the instance.
        We need to know both for later in the pipeline. -->
  <xsl:variable name="StartAtInstString">
    <xsl:choose>
      <xsl:when test="$start_at_inst != ''">
        <xsl:message>*** StartAtInstString: Using $start_at_inst</xsl:message>
        <xsl:value-of select="$start_at_inst"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:message>*** StartAtInstString: Falling back on generating path</xsl:message>
        <xsl:apply-templates select="$StartAtRNGNodeSet" mode="generatePathMode"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  
  <xsl:template match="*|@*" mode="generatePathMode">
    <xsl:call-template name="RNGPathToSelf"/>
  </xsl:template>
  
  <!--**************************************************************-->
  <!-- Templates                                                    -->
  <!--**************************************************************-->
 
  <xsl:template match="grammar">
    <grammar>
      <xsl:if test="not(boolean($StartAtRNGNodeSet))"> <!--assert-->
        <xsl:message terminate="yes">*** ERROR $StartAtRNGNodeSet is empty!</xsl:message>
      </xsl:if>
      <rngsub-start-at-rng><xsl:value-of select="$StartAtRNGString"/></rngsub-start-at-rng>
      <rngsub-start-at-inst><xsl:value-of select="$StartAtInstString"/></rngsub-start-at-inst>
      <rngsub-start-at-title>
        <xsl:value-of select="substring-after($StartAtInstString, '/')"/>
      </rngsub-start-at-title>

      <xsl:apply-templates select="$StartAtRNGNodeSet"/>

    </grammar>
  </xsl:template>

  <!--**************************************************************-->
  <!--  Truncate the input tree of RNG grammar at a certain depth
        The depth is dynamically determined by the presence of a
        zeroOrMore or a oneOrMore element. The subtree of those elements
        is truncated. Performance. -->
  <xsl:template match="*|@*">
    <xsl:choose>
      <xsl:when test="ancestor::zeroOrMore or ancestor::oneOrMore">
        <xsl:choose>
          <xsl:when
            test="$StartAtRNGNodeSet//zeroOrMore//*[generate-id()=generate-id(current())]
                or $StartAtRNGNodeSet//oneOrMore//*[generate-id()=generate-id(current())]
            ">
            <rngsub_stopped>
              <rng_path>
                <xsl:call-template name="thePathTemplate"/>
              </rng_path>
              <xsl:text> </xsl:text>
              <inst_path>
                <xsl:call-template name="RNGPathToSelf"/>
              </inst_path>
            </rngsub_stopped>
          </xsl:when>
          <xsl:otherwise>
            <xsl:call-template name="simpleRecurseTemplate"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="simpleRecurseTemplate"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <!--**************************************************************-->
  <!-- Named Templates                                              -->
  <!--**************************************************************-->

  <xsl:template name="simpleRecurseTemplate">
    <xsl:copy>
      <xsl:apply-templates select="node()|@*"/>
    </xsl:copy>
  </xsl:template>

  <xsl:template name="RNGPathToSelf">
    <xsl:for-each select="current()/ancestor-or-self::*[@name]">
      <xsl:text>/</xsl:text>
      <xsl:value-of select="@name"/>
    </xsl:for-each>
    <!--ancestor-or-self::*/@name is broke in the version of libXSLT i'm using-->
  </xsl:template>
  
  <xsl:template match="define"/>
  
  <!--**************************************************************-->
  <!--  Convert an instance XPath to an RNG XPath. -->
  <xsl:template name="inst2rngPath">
    <xsl:param name="fragment"/>
    <xsl:choose>
      <xsl:when test="starts-with($fragment, '/')">  <!-- strip beginning / -->
        <xsl:call-template name="inst2rngPath">
          <xsl:with-param name="fragment" select="substring-after($fragment, '/')"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:text>//*[@name='</xsl:text>
        <xsl:choose><xsl:when test="substring-before($fragment, '/')">
            <xsl:value-of select="substring-before($fragment, '/')"/>
          </xsl:when><xsl:otherwise>
            <xsl:value-of select="$fragment"/>
        </xsl:otherwise></xsl:choose>
        <xsl:text>']</xsl:text>
        <xsl:choose>
          <xsl:when test="contains($fragment, '/')">  <!-- recursive step -->
            <xsl:call-template name="inst2rngPath">
              <xsl:with-param name="fragment" select="substring-after($fragment, '/')"/>
            </xsl:call-template>
          </xsl:when><xsl:otherwise>  <!-- default step -->
            <xsl:text></xsl:text>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
 
  <!--**************************************************************-->
  <!--  Some Magic from Dimitre Novatchev that generates the XPath
        of the context node! -->
  <xsl:template name="thePathTemplate">
    <xsl:variable name="thePathVar">
      <xsl:variable name="theNode" select="."/>
      <xsl:for-each select="$theNode | $theNode/ancestor-or-self::node()[..]">
        <xsl:element name="slash">/</xsl:element>
        <xsl:choose>
          <xsl:when test="self::*">     
            <xsl:element name="nodeName">
              <xsl:value-of select="name()"/>
              <xsl:variable name="thisPosition" 
                  select="count(preceding-sibling::*[name(current()) = name()])"/>
              <xsl:variable name="numFollowing" 
                  select="count(following-sibling::*[name(current()) = name()])"/>
              <xsl:if test="$thisPosition + $numFollowing > 0">
                <xsl:value-of select="concat('[', $thisPosition + 1, ']')"/>
              </xsl:if>
            </xsl:element>
          </xsl:when>
          <xsl:otherwise> <!-- This node is not an element -->
            <xsl:choose>
              <xsl:when test="count(. | ../@*) = count(../@*)"> 
                <!-- Attribute -->
                <xsl:element name="nodeName">
                  <xsl:value-of select="concat('@',name())"/>
                </xsl:element>
              </xsl:when>   
              <xsl:when test="self::text()">  
                <!-- Text -->
                <xsl:element name="nodeName">
                  <xsl:value-of select="'text()'"/>
                  <xsl:variable name="thisPosition" 
                      select="count(preceding-sibling::text())"/>
                  <xsl:variable name="numFollowing" 
                      select="count(following-sibling::text())"/>
                  <xsl:if test="$thisPosition + $numFollowing > 0">
                    <xsl:value-of select="concat('[', $thisPosition + 1, ']')"/>
                  </xsl:if>
                </xsl:element>
              </xsl:when>   
              <xsl:when test="self::processing-instruction()">
                <!-- Processing Instruction -->
                <xsl:element name="nodeName">
                  <xsl:value-of select="'processing-instruction()'"/>
                  <xsl:variable name="thisPosition" 
                     select="count(preceding-sibling::processing-instruction())"/>
                  <xsl:variable name="numFollowing" 
                      select="count(following-sibling::processing-instruction())"/>
                  <xsl:if test="$thisPosition + $numFollowing > 0">
                    <xsl:value-of select="concat('[', $thisPosition + 1, ']')"/>
                  </xsl:if>
                </xsl:element>
              </xsl:when>   
              <xsl:when test="self::comment()"> 
                <!-- Comment -->
                <xsl:element name="nodeName">
                  <xsl:value-of select="'comment()'"/>
                  <xsl:variable name="thisPosition" 
                      select="count(preceding-sibling::comment())"/>
                  <xsl:variable name="numFollowing" 
                      select="count(following-sibling::comment())"/>
                  <xsl:if test="$thisPosition + $numFollowing > 0">
                    <xsl:value-of select="concat('[', $thisPosition + 1, ']')"/>
                  </xsl:if>
                </xsl:element>
              </xsl:when>   
              <xsl:when test="count(. | ../namespace::*) = count(../namespace::*)">
                <!-- Namespace: -->
                <xsl:variable name="apos">'</xsl:variable>
                <xsl:element name="nodeName">
                  <xsl:value-of select="concat('namespace::*', 
                  '[local-name() = ', $apos, local-name(), $apos, ']')"/>
                  </xsl:element>
                </xsl:when>   
              </xsl:choose>
            </xsl:otherwise>      
          </xsl:choose>
        </xsl:for-each>
      </xsl:variable>
    <xsl:value-of select="$thePathVar"/>
  </xsl:template>

</xsl:stylesheet>

